/* icp_from_file.cpp
 * Align multiple point clouds in one 3D model. To run, change numClouds variable to the number of pcd files you are inputting.
 * Right now all of the variables are set to fit the pcds in cluttered_desk.
 * Input clouds should be "inputCloud#.pcd" where # is a number.
 * Created by 3d-scanning on the cheap
 */
#include <stdio.h>

#include <queue>

#include <pcl/io/pcd_io.h>
#include <pcl/io/openni_grabber.h>
#include <pcl/visualization/cloud_viewer.h>
#include <pcl/point_types.h>
#include <pcl/registration/icp.h>
#include <pcl/registration/gicp.h>
#include <pcl/filters/filter.h>
#include <pcl/keypoints/sift_keypoint.h>
#include <pcl/common/transforms.h>
#include <pcl/features/normal_3d.h>
#include <pcl/filters/passthrough.h>
#include <pcl/io/ply_io.h>

class icp_from_file
{
public:
	void run ()
	{
		printf("Starting... \n");
		//Initialize objects
		viewer = new pcl::visualization::CloudViewer ("ICP Viewer");
		finalModel = new pcl::PointCloud<pcl::PointXYZRGBA>();
		XYZRGBAtarget = new pcl::PointCloud<pcl::PointXYZRGBA>();
		cumulativeTransform = new Eigen::Matrix4f();
		*cumulativeTransform = Eigen::Matrix4f::Identity();
		icp = new pcl::IterativeClosestPointWithNormals<pcl::PointNormal, pcl::PointNormal>;
		icp->setMaximumIterations(50);
		icp->setMaxCorrespondenceDistance(1);
		icp->setTransformationEpsilon (1e-6);
		
		//Load pcd files into queue
		std::queue<pcl::PointCloud<pcl::PointXYZRGBA>::Ptr> cloudQ;
		cloudQ = *loadFiles(12);
		//Align each cloud in queue into final model via ICP
		int count = 0;
		while(cloudQ.size() > 0){
			printf("Iteration number %i \n", count);
			doICP(cloudQ.front());
			cloudQ.pop();
			count++;
		}
		view(finalModel->makeShared());
		pcl::io::savePCDFile("3dmodel.pcd", *finalModel);
		printf("All done \n");
		
		// keeps the callback from exiting too quickly
		while (! viewer->wasStopped())
		{
			boost::this_thread::sleep (boost::posix_time::seconds (1));
		}
		printf("ending run()\n");
	}

private:
	pcl::visualization::CloudViewer* viewer; //Used for viewing point clouds
	pcl::IterativeClosestPointWithNormals<pcl::PointNormal, pcl::PointNormal>* icp;
	pcl::PointCloud<pcl::PointXYZRGBA>* finalModel; //Stores clouds successively aligned via ICP
	pcl::PointCloud<pcl::PointXYZRGBA>* XYZRGBAtarget; //Stores target alignment cloud in XYZRGBA form. Useful for viewing/debugging.
	Eigen::Matrix4f* cumulativeTransform; //Store cumulative transformation needed to align each cloud to final model

	//Loads .pcd files as returns them in a queue. numClouds is the number of clouds to load, inclusive
	std::queue<pcl::PointCloud<pcl::PointXYZRGBA>::Ptr>* loadFiles(int numClouds){
		std::queue<pcl::PointCloud<pcl::PointXYZRGBA>::Ptr>* cloudQ (new std::queue<pcl::PointCloud<pcl::PointXYZRGBA>::Ptr>);
		for(int i=0;i<=numClouds;i++){
			pcl::PointCloud<pcl::PointXYZRGBA>::Ptr curCloud (new pcl::PointCloud<pcl::PointXYZRGBA>);
			std::string filename ("inputCloud" + std::to_string(static_cast<long long>(i)) + ".pcd");
			if(pcl::io::loadPCDFile<pcl::PointXYZRGBA> (filename, *curCloud) == -1){
				printf("Couldn't load file: %s\n", filename);
				return cloudQ;
			}
			//Apply bounding box early to save memory
			*curCloud = *filterBounds(curCloud, 2); // 2 is good for cluttered desk
			cloudQ->push(curCloud);
		}
		return cloudQ;
	}

	void doICP(pcl::PointCloud<pcl::PointXYZRGBA>::Ptr source){
		printf("Preparing for ICP... \n");

		//Remove NaNs
		printf("Cleaning source... \n");
		pcl::PointCloud<pcl::PointXYZRGBA>::Ptr cleanSource (new pcl::PointCloud<pcl::PointXYZRGBA>);
		*cleanSource = *filterNaNs(source);

		//Downsample to reduce number of points
		printf("Downsampling source... \n");
		pcl::PointCloud<pcl::PointXYZRGBA>::Ptr downsampledSource (new pcl::PointCloud<pcl::PointXYZRGBA>);
		*downsampledSource = *downsample(cleanSource, 0.005f);

		//Calculate normals
		printf("Calculating source normals... \n");
		pcl::PointCloud<pcl::PointNormal>::Ptr normalsSource (new pcl::PointCloud<pcl::PointNormal>);
		*normalsSource = *findNormals(downsampledSource);
		
		printf("Size of raw cloud: %i \n", cleanSource->points.size());
		printf("Size of downsampled cloud: %i \n", normalsSource->points.size());

		//If this is first iteration, just set target cloud to input
		if(! icp->getInputTarget()){
			printf("First iteration \n");
			*finalModel += *downsampledSource;
			icp->setInputTarget(normalsSource);
			printf("Set target \n");
			view(finalModel->makeShared());
			printf("Press enter to continue aligning \n");
			cin.get();
			return;
		}

		//Calculate transformation to align incoming source cloud to target
		icp->setInputSource(normalsSource);
		pcl::PointCloud<pcl::PointNormal> throwaway; //We can ignore the transformed source cloud since we apply the transformation manually
		printf("Running ICP... \n");
		icp->align(throwaway);
		printf("fitness score: %f\n", icp->getFitnessScore()); //As it turns out, this is not a useful metric
		
		//Apply transformation to source cloud to align it with target
		printf("Applying transformation... \n");
		pcl::PointCloud<pcl::PointXYZRGBA>::Ptr transformed (new pcl::PointCloud<pcl::PointXYZRGBA>);
		pcl::transformPointCloud(*cleanSource, *transformed, icp->getFinalTransformation());
		
		//If you are wanting to view each step along the way, then these lines can be uncommented
		//printf("Showing aligned source \n");
		//view(transformed);

		//View pairwise alignment of just target and source to make sure it's accurate
		//pcl::PointCloud<pcl::PointXYZRGBA>::Ptr targetAndSource (new pcl::PointCloud<pcl::PointXYZRGBA>);
		//*targetAndSource = *transformed + *XYZRGBAtarget; 
		//printf("Showing aligned source + target \n");
		//view(targetAndSource);

		//Apply cumulative transformation to source to align it to cumulative model
		//And update cumulative transformation
		pcl::transformPointCloud(*transformed, *transformed, *cumulativeTransform);
		*cumulativeTransform = *cumulativeTransform * icp->getFinalTransformation(); //Update cumulative transformation
		printf("Updated cumulative transformation matrix: \n");
		//Add to model
		printf("Updating finalModel... \n");
		*finalModel += *transformed;
		*finalModel = *downsample(finalModel->makeShared(), 0.005f);
		printf("Size of updated final model: %i \n", finalModel->points.size());
		
		printf("Showing updated model \n");
		view(finalModel->makeShared());
			
		icp->setInputTarget(normalsSource); //Set source cloud to target in preparation for next iteration
		//*XYZRGBAtarget = *cleanSource; //Also update XYZRGBA target for viewing later
	}

	//Returns a downsampled version of input cloud to reduce number of points
	//Inspired by http://www.pcl-users.org/file/n4019799/Main.cpp
	pcl::PointCloud<pcl::PointXYZRGBA>::Ptr downsample (pcl::PointCloud<pcl::PointXYZRGBA>::Ptr cloud, const float leaf){
		pcl::VoxelGrid<pcl::PointXYZRGBA> vox;
		pcl::PointCloud<pcl::PointXYZRGBA>::Ptr downsampled (new pcl::PointCloud<pcl::PointXYZRGBA>);
		vox.setInputCloud(cloud);
		vox.setLeafSize(leaf, leaf, leaf); // 0.005f good for cluttered desk - smaller = more points
		vox.filter(*downsampled);
		return downsampled;
	}
	
	//Returns a cloud with normals calculated from input cloud
	//Inspired by http://pointclouds.org/documentation/tutorials/pairwise_incremental_registration.php
	pcl::PointCloud<pcl::PointNormal>::Ptr findNormals (pcl::PointCloud<pcl::PointXYZRGBA>::Ptr cloud){
		pcl::NormalEstimation<pcl::PointXYZRGBA, pcl::PointNormal> normEst;
		pcl::search::KdTree<pcl::PointXYZRGBA>::Ptr tree (new pcl::search::KdTree<pcl::PointXYZRGBA> ());
		normEst.setSearchMethod(tree);
		normEst.setKSearch(30); //Search for 30 neighbors

		normEst.setInputCloud(cloud);
		pcl::PointCloud<pcl::PointNormal>::Ptr withNormals (new pcl::PointCloud<pcl::PointNormal>);
		normEst.compute(*withNormals);
		pcl::copyPointCloud(*cloud, *withNormals);
		return withNormals;
	}

	//Removes points outside of a certain distance from the camera
	pcl::PointCloud<pcl::PointXYZRGBA>::Ptr filterBounds(pcl::PointCloud<pcl::PointXYZRGBA>::Ptr cloud, double size){
		//Initialize filter and cloud to return
		pcl::PassThrough<pcl::PointXYZRGBA> pass;
		pass.setInputCloud(cloud);

		// Need a new cloud instead of modifying in place
		// because of a PCL bug that only affects windows
		pcl::PointCloud<pcl::PointXYZRGBA>::Ptr filtered (new pcl::PointCloud<pcl::PointXYZRGBA>);

		//Filter x coordinate
		pass.setFilterFieldName("x");
		pass.setFilterLimits(-size, size);
		pass.filter(*filtered);

		//Filter y coordinate
		pass.setFilterFieldName("y");
		pass.setFilterLimits(1-size, 1+size);
		pass.filter(*filtered);

		//Filter z coordinate
		pass.setFilterFieldName("z");
		pass.setFilterLimits(0, size);
		pass.filter(*filtered);
		return filtered;
	}

	//Displays given cloud in the viewer
	void view(pcl::PointCloud<pcl::PointXYZRGBA>::Ptr cloud){
		viewer->showCloud(cloud);
	}

	//Remove NaN values from a given cloud
	pcl::PointCloud<pcl::PointXYZRGBA>::Ptr filterNaNs (pcl::PointCloud<pcl::PointXYZRGBA>::Ptr dirtyCloud){
			std::vector<int> indices; //This is used for mapping point indeces after NaN cleaning, we don't really need it
			pcl::PointCloud<pcl::PointXYZRGBA>::Ptr cleanCloud (new pcl::PointCloud<pcl::PointXYZRGBA>);
			pcl::removeNaNFromPointCloud(*dirtyCloud, *cleanCloud, indices);
			return cleanCloud;
	}
};

int main ()
{
	icp_from_file fromFile;
	fromFile.run ();
	return 0;
}